"""Instantiate the OpenMC Surfaces needed by the core model.

The geometric parameters defining the core model are tabulated here.
The geometric specifications are loosely based upon NuScale's Small
Modular Pressurized Water Reactor concept as detailed here in their
NRC design certification (DC) documentation:

https://www.nrc.gov/docs/ML1618/ML16187A017.pdf
https://www.nrc.gov/docs/ML1700/ML17007A001.pdf

NuScale DC application, chapter 1: https://www.nrc.gov/docs/ML1701/ML17013A264.pdf
NuScale DC application, chapter 4: https://www.nrc.gov/docs/ML1701/ML17013A274.pdf

"""

from math import tan, pi

import numpy as np
import openmc

# INCHES = 2.54

# Notation
# GT: Guide Tube
# BA: Burnable Absorber
# CP: Control Poison
# FR: Fuel Rod
# IR: Inner Radius
# OR: Outer Radius
# IT: Instrument Tube
# FA: Fuel Assembly
# RPV: Reactor Pressure Vessel

# Radial dimensions of pins
fuel_OR = 0.4058
clad_IR = 0.4140
clad_OR = 0.4750
inconel_OR = 0.0646
guide_tube_IR = 0.5715
guide_tube_OR = 0.6121
aic_OR = 0.4267
b4c_OR = 0.4229
cr_clad_IR = 0.4369
cr_clad_OR = 0.4839

core_barrel_IR = 93.9800
core_barrel_OR = 99.0600

spacer_grid_inner_pitch = 1.2242
pin_cell_pitch = 1.2598
# pin_cell_pitch = 1.261
assembly_pitch = 21.5036

# Axial dimensions
bottom_nozzle_bottom  = 0.
bottom_nozzle_top     = 10.160
end_cap_lower_bottom  = bottom_nozzle_top
end_cap_lower_top     = end_cap_lower_bottom + 1.205
fuel_bottom           = end_cap_lower_top
fuel_top              = fuel_bottom + 200.
plenum_spring_bottom  = fuel_top
plenum_spring_top     = plenum_spring_bottom + 13.490
end_cap_upper_bottom  = plenum_spring_top
end_cap_upper_top     = end_cap_upper_bottom + 1.205
coolant_upper_bottom  = end_cap_upper_top
coolant_upper_top     = coolant_upper_bottom + 8.481
top_nozzle_bottom     = coolant_upper_top
top_nozzle_top        = top_nozzle_bottom + 9.020

cr_ins_empty_gt_bottom     = bottom_nozzle_top
cr_ins_empty_gt_top        = cr_ins_empty_gt_bottom + 8.386
cr_ins_end_plug_bottom     = cr_ins_empty_gt_top
cr_ins_end_plug_top        = cr_ins_end_plug_bottom + 4.859
cr_ins_aic_bottom          = cr_ins_end_plug_top
cr_ins_aic_top             = cr_ins_aic_bottom + 30.480
cr_ins_b4c_bottom          = cr_ins_aic_top
cr_ins_b4c_top             = cr_ins_b4c_bottom + 157.480
cr_ins_upper_plenum_bottom = cr_ins_b4c_top
cr_ins_upper_plenum_top    = cr_ins_b4c_top + 23.176

cr_ej_end_plug_bottom = bottom_nozzle_top + 196.346
cr_ej_end_plug_top    = cr_ej_end_plug_bottom + 4.859
cr_ej_aic_bottom      = cr_ej_end_plug_top
cr_ej_aic_top         = cr_ej_aic_bottom + 23.176

sg1_bottom = 14.920
sg1_top = 19.365
sg2_bottom = 65.974
sg2_top = 70.419
sg3_bottom = 117.028
sg3_top = 121.473
sg4_bottom = 168.082
sg4_top = 172.527
sg5_bottom = 219.136
sg5_top = 223.581

surfs = {}

# Cylinders
surfs['Fuel OR'] = openmc.ZCylinder(r=fuel_OR)
surfs['Cladding IR'] = openmc.ZCylinder(r=clad_IR)
surfs['Cladding OR'] = openmc.ZCylinder(r=clad_OR)
surfs['Inconel OR']  = openmc.ZCylinder(r=inconel_OR)
surfs['GT IR'] = openmc.ZCylinder(r=guide_tube_IR)
surfs['GT OR'] = openmc.ZCylinder(r=guide_tube_OR)
surfs['AIC OR'] = openmc.ZCylinder(r=aic_OR)
surfs['B4C OR'] = openmc.ZCylinder(r=b4c_OR)
surfs['CR cladding IR'] = openmc.ZCylinder(r=cr_clad_IR)
surfs['CR cladding OR'] = openmc.ZCylinder(r=cr_clad_OR)

surfs['core barrel IR'] = openmc.ZCylinder(r=core_barrel_IR)
# surfs['core barrel IR'].boundary_type = 'reflective'
surfs['core barrel OR'] = openmc.ZCylinder(r=core_barrel_OR)
surfs['core barrel OR'].boundary_type = 'vacuum'

# Rectangular prisms for grid spacers
surfs['spacer grid box in'] = openmc.model.RectangularPrism(spacer_grid_inner_pitch, spacer_grid_inner_pitch)
surfs['spacer grid box out'] = openmc.model.RectangularPrism(pin_cell_pitch, pin_cell_pitch)

# Horizontal planes
surfs['bottom nozzle bottom'] = openmc.ZPlane(z0=bottom_nozzle_bottom)
surfs['bottom nozzle top'] = openmc.ZPlane(z0=bottom_nozzle_top)
surfs['end cap lower bottom'] = openmc.ZPlane(z0=end_cap_lower_bottom)
surfs['end cap lower top'] = openmc.ZPlane(z0=end_cap_lower_top)
surfs['fuel bottom'] = openmc.ZPlane(z0=fuel_bottom)
surfs['fuel top'] = openmc.ZPlane(z0=fuel_top)
surfs['plenum spring bottom'] = openmc.ZPlane(z0=plenum_spring_bottom)
surfs['plenum spring top'] = openmc.ZPlane(z0=plenum_spring_top)
surfs['end cap upper bottom'] = openmc.ZPlane(z0=end_cap_upper_bottom)
surfs['end cap upper top'] = openmc.ZPlane(z0=end_cap_upper_top)
surfs['coolant upper bottom'] = openmc.ZPlane(z0=coolant_upper_bottom)
surfs['coolant upper top'] = openmc.ZPlane(z0=coolant_upper_top)
surfs['top nozzle bottom'] = openmc.ZPlane(z0=top_nozzle_bottom)
surfs['top nozzle top'] = openmc.ZPlane(z0=top_nozzle_top)

surfs['cr ins empty gt bottom'] = openmc.ZPlane(z0=cr_ins_empty_gt_bottom)
surfs['cr ins empty gt top'] = openmc.ZPlane(z0=cr_ins_empty_gt_top)
surfs['cr ins end plug bottom'] = openmc.ZPlane(z0=cr_ins_end_plug_bottom)
surfs['cr ins end plug top'] = openmc.ZPlane(z0=cr_ins_end_plug_top)
surfs['cr ins aic bottom'] = openmc.ZPlane(z0=cr_ins_aic_bottom)
surfs['cr ins aic top'] = openmc.ZPlane(z0=cr_ins_aic_top)
surfs['cr ins b4c bottom'] = openmc.ZPlane(z0=cr_ins_b4c_bottom)
surfs['cr ins b4c top'] = openmc.ZPlane(z0=cr_ins_b4c_top)
surfs['cr ins upper plenum bottom'] = openmc.ZPlane(z0=cr_ins_upper_plenum_bottom)
surfs['cr ins upper plenum top'] = openmc.ZPlane(z0=cr_ins_upper_plenum_top)

surfs['cr ej end plug bottom'] = openmc.ZPlane(z0=cr_ej_end_plug_bottom)
surfs['cr ej end plug top'] = openmc.ZPlane(z0=cr_ej_end_plug_top)
surfs['cr ej aic bottom'] = openmc.ZPlane(z0=cr_ej_aic_bottom)
surfs['cr ej aic top'] = openmc.ZPlane(z0=cr_ej_aic_top)

surfs['sg1 bottom'] = openmc.ZPlane(z0=sg1_bottom)
surfs['sg1 top'] = openmc.ZPlane(z0=sg1_top)
surfs['sg2 bottom'] = openmc.ZPlane(z0=sg2_bottom)
surfs['sg2 top'] = openmc.ZPlane(z0=sg2_top)
surfs['sg3 bottom'] = openmc.ZPlane(z0=sg3_bottom)
surfs['sg3 top'] = openmc.ZPlane(z0=sg3_top)
surfs['sg4 bottom'] = openmc.ZPlane(z0=sg4_bottom)
surfs['sg4 top'] = openmc.ZPlane(z0=sg4_top)
surfs['sg5 bottom'] = openmc.ZPlane(z0=sg5_bottom)
surfs['sg5 top'] = openmc.ZPlane(z0=sg5_top)

surfs['lower bound'] = surfs['bottom nozzle bottom']
surfs['lower bound'].boundary_type = 'vacuum'
surfs['upper bound'] = surfs['top nozzle top']
surfs['upper bound'].boundary_type = 'vacuum'
# surfs['bottom'] = surfs['core bottom']
# surfs['top'] = surfs['top nozzle top']
